// config/database.js - Configuração do MySQL
const mysql = require('mysql2/promise');

// Configuração do pool de conexões
const pool = mysql.createPool({
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'wwsist_valdecir',
    password: process.env.DB_PASSWORD || 'Valdecir2509*',
    database: process.env.DB_NAME || 'wwsist_educa',
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0,
    charset: 'utf8mb4',
    timezone: '-03:00' // Horário de Brasília
});

// Função para testar conexão
async function testConnection() {
    try {
        const connection = await pool.getConnection();
        await connection.ping();
        connection.release();
        return true;
    } catch (error) {
        console.error('Erro ao conectar com banco de dados:', error.message);
        throw error;
    }
}

// Função para executar queries
async function query(sql, params = []) {
    try {
        const [rows] = await pool.execute(sql, params);
        return rows;
    } catch (error) {
        console.error('Erro na query:', error.message);
        console.error('SQL:', sql);
        console.error('Params:', params);
        throw error;
    }
}

// Função para executar queries com transação
async function transaction(queries) {
    const connection = await pool.getConnection();
    
    try {
        await connection.beginTransaction();
        
        const results = [];
        for (const { sql, params } of queries) {
            const [result] = await connection.execute(sql, params);
            results.push(result);
        }
        
        await connection.commit();
        return results;
    } catch (error) {
        await connection.rollback();
        throw error;
    } finally {
        connection.release();
    }
}

// Função para buscar um registro
async function findOne(table, conditions = {}, select = '*') {
    const whereClause = Object.keys(conditions).length > 0 
        ? 'WHERE ' + Object.keys(conditions).map(key => `${key} = ?`).join(' AND ')
        : '';
    
    const sql = `SELECT ${select} FROM ${table} ${whereClause} LIMIT 1`;
    const params = Object.values(conditions);
    
    const rows = await query(sql, params);
    return rows.length > 0 ? rows[0] : null;
}

// Função para buscar múltiplos registros
async function findMany(table, conditions = {}, options = {}) {
    const {
        select = '*',
        orderBy = '',
        limit = '',
        offset = ''
    } = options;
    
    const whereClause = Object.keys(conditions).length > 0 
        ? 'WHERE ' + Object.keys(conditions).map(key => `${key} = ?`).join(' AND ')
        : '';
    
    const orderClause = orderBy ? `ORDER BY ${orderBy}` : '';
    const limitClause = limit ? `LIMIT ${limit}` : '';
    const offsetClause = offset ? `OFFSET ${offset}` : '';
    
    const sql = `SELECT ${select} FROM ${table} ${whereClause} ${orderClause} ${limitClause} ${offsetClause}`;
    const params = Object.values(conditions);
    
    return await query(sql, params);
}

// Função para inserir registro
async function insert(table, data) {
    const columns = Object.keys(data).join(', ');
    const placeholders = Object.keys(data).map(() => '?').join(', ');
    const values = Object.values(data);
    
    const sql = `INSERT INTO ${table} (${columns}) VALUES (${placeholders})`;
    const result = await query(sql, values);
    
    return {
        insertId: result.insertId,
        affectedRows: result.affectedRows
    };
}

// Função para atualizar registro
async function update(table, data, conditions) {
    const setClause = Object.keys(data).map(key => `${key} = ?`).join(', ');
    const whereClause = Object.keys(conditions).map(key => `${key} = ?`).join(' AND ');
    
    const sql = `UPDATE ${table} SET ${setClause} WHERE ${whereClause}`;
    const params = [...Object.values(data), ...Object.values(conditions)];
    
    const result = await query(sql, params);
    return {
        affectedRows: result.affectedRows,
        changedRows: result.changedRows
    };
}

// Função para deletar registro
async function deleteRecord(table, conditions) {
    const whereClause = Object.keys(conditions).map(key => `${key} = ?`).join(' AND ');
    const sql = `DELETE FROM ${table} WHERE ${whereClause}`;
    const params = Object.values(conditions);
    
    const result = await query(sql, params);
    return {
        affectedRows: result.affectedRows
    };
}

// Função para contar registros
async function count(table, conditions = {}) {
    const whereClause = Object.keys(conditions).length > 0 
        ? 'WHERE ' + Object.keys(conditions).map(key => `${key} = ?`).join(' AND ')
        : '';
    
    const sql = `SELECT COUNT(*) as total FROM ${table} ${whereClause}`;
    const params = Object.values(conditions);
    
    const result = await query(sql, params);
    return result[0].total;
}

// Função para paginação
async function paginate(table, conditions = {}, options = {}) {
    const {
        select = '*',
        orderBy = 'id DESC',
        page = 1,
        perPage = 10
    } = options;
    
    const offset = (page - 1) * perPage;
    
    // Buscar registros
    const data = await findMany(table, conditions, {
        select,
        orderBy,
        limit: perPage,
        offset
    });
    
    // Contar total de registros
    const total = await count(table, conditions);
    
    return {
        data,
        pagination: {
            page: parseInt(page),
            perPage: parseInt(perPage),
            total,
            totalPages: Math.ceil(total / perPage),
            hasNext: page < Math.ceil(total / perPage),
            hasPrev: page > 1
        }
    };
}

// Função para executar query personalizada de busca
async function search(table, searchTerm, searchColumns, options = {}) {
    const {
        select = '*',
        conditions = {},
        orderBy = 'id DESC',
        limit = 50
    } = options;
    
    // Construir cláusula de busca
    const searchClause = searchColumns
        .map(column => `${column} LIKE ?`)
        .join(' OR ');
    
    // Construir cláusula WHERE
    const conditionsClause = Object.keys(conditions).length > 0 
        ? Object.keys(conditions).map(key => `${key} = ?`).join(' AND ')
        : '';
    
    const whereClause = [
        searchClause ? `(${searchClause})` : '',
        conditionsClause
    ].filter(Boolean).join(' AND ');
    
    const sql = `
        SELECT ${select} 
        FROM ${table} 
        ${whereClause ? `WHERE ${whereClause}` : ''}
        ORDER BY ${orderBy}
        LIMIT ${limit}
    `;
    
    // Preparar parâmetros
    const searchParams = searchColumns.map(() => `%${searchTerm}%`);
    const conditionParams = Object.values(conditions);
    const params = [...searchParams, ...conditionParams];
    
    return await query(sql, params);
}

// Função para obter estatísticas
async function getStats() {
    const stats = {};
    
    try {
        // Estatísticas de usuários
        const userStats = await query(`
            SELECT 
                role,
                COUNT(*) as count,
                COUNT(CASE WHEN active = 1 THEN 1 END) as active_count
            FROM users 
            GROUP BY role
        `);
        
        stats.users = userStats.reduce((acc, row) => {
            acc[row.role] = {
                total: row.count,
                active: row.active_count
            };
            return acc;
        }, {});
        
        // Estatísticas de escolas
        const schoolStats = await query(`
            SELECT 
                COUNT(*) as total_schools,
                COUNT(CASE WHEN active = 1 THEN 1 END) as active_schools
            FROM schools
        `);
        
        stats.schools = schoolStats[0];
        
        // Estatísticas de conversas
        const conversationStats = await query(`
            SELECT 
                type,
                COUNT(*) as count
            FROM conversations 
            WHERE active = 1
            GROUP BY type
        `);
        
        stats.conversations = conversationStats.reduce((acc, row) => {
            acc[row.type] = row.count;
            return acc;
        }, {});
        
        // Estatísticas de mensagens (últimos 30 dias)
        const messageStats = await query(`
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as count
            FROM messages 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        `);
        
        stats.messages = messageStats;
        
        return stats;
    } catch (error) {
        console.error('Erro ao obter estatísticas:', error);
        return {};
    }
}

// Função para limpar dados antigos
async function cleanup() {
    try {
        // Limpar mensagens muito antigas (opcional - comentado por padrão)
        // await query(`
        //     DELETE FROM messages 
        //     WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 YEAR)
        // `);
        
        // Limpar notificações WhatsApp antigas
        await query(`
            DELETE FROM whatsapp_notifications 
            WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)
        `);
        
        console.log('✅ Limpeza de dados concluída');
    } catch (error) {
        console.error('❌ Erro na limpeza de dados:', error);
    }
}

// Executar limpeza a cada 24 horas
setInterval(cleanup, 24 * 60 * 60 * 1000);

module.exports = {
    pool,
    query,
    transaction,
    findOne,
    findMany,
    insert,
    update,
    deleteRecord,
    count,
    paginate,
    search,
    testConnection,
    getStats,
    cleanup
};